#include "EffectsManager.h"

EffectsManager :: EffectsManager() {
   memset(emitterUsed,0,MAX_EMITTER_COUNT);
}

EffectsManager :: ~EffectsManager() {
}

EffectsManager* EffectsManager :: instance() {
   static EffectsManager *e = new EffectsManager;
   return e;
}

void EffectsManager :: init() {
   initTextures();
}

void EffectsManager :: initTextures() {

   expDecalTex = new Texture("data/brightdot2.bmp");
   expDecalTex->build();

   bloodSplatTex = new Texture("data/blood.bmp");
   bloodSplatTex->build();

   yellowSmokeTex = new Texture("data/smoke2.tga");
   yellowSmokeTex->build();

   explosionShader = new Texture("data/explosion.shader");
   explosionShader->build();

   pexpTex = new Texture("data/p3.tga");
   pexpTex->build();

   for (int i=0;i<5;++i) {
      Texture *t = new Texture("data/explosion.shader");
      t->build();
      freeExpTextures.push_back(t);
   }
  
   createPlayerDamageEmitters();
   createPlayerExplosionEmitters();

   missileTrailTemplate = new Particle("data/msltrail.part");

}

void EffectsManager :: reset() {

   removeEmitter(peid[0]);
   removeEmitter(peid[1]);
   createPlayerDamageEmitters();

}

void EffectsManager :: makeExplosion(float x,float y) {
   
   tExplosion explosion;
   explosion.x = x;
   explosion.y = y;
   explosion.time = 1.0f;
   
   if (freeExpTextures.size() > 0) {
      explosion.tex = freeExpTextures.front();
      freeExpTextures.pop_front();
   }
   else {
      explosion.tex = explosionShader;
   }   
      
   explosion.tex->resetShaders();
   explosions.push_back(explosion);

}

void EffectsManager :: makePlayerExplosion(int pid) {
   emitters[pexeid[pid]].active = true;
   emitters[pexeid[pid]].timed = true;
   emitters[pexeid[pid]].timeout = 0.5f;
   emitters[pexeid[pid]].emitter->setCenter(Vector3d(px[pid],py[pid],0));
   emitters[pexeid[pid]].emitter->addRandomEntities();
}

void EffectsManager :: createExplosionDecal(float x,float y,float life_scale) {

   tExpDecal decal;
   decal.x = x;
   decal.y = y;
   decal.life_scale = life_scale;
   decal.remaining_life = 1.0f;

   expDecals.push_back(decal);

}

void EffectsManager :: createBloodSplat(float x,float y,float life_scale) {

   tBloodSplat decal;
   decal.x = x;
   decal.y = y;
   decal.life_scale = life_scale;
   decal.remaining_life = 1.0f;
   decal.rot = rand();

   bloodSplats.push_back(decal);

}

int  EffectsManager :: createMissileTrail(float x,float y) {

   int eid = getFirstFreeEmitter();
   if (eid == -1)
      return -1;

   tEmitter &e = emitters[eid];
   e.eid = eid;
   e.active = true;
   e.timed = false;
   e.killme = false;
   e.emitter = new Particle;
   missileTrailTemplate->copyData(e.emitter);
   e.emitter->setCenter(Vector3d(x,y,0));
   e.emitter->reset();

   return e.eid;
}

void EffectsManager :: removeEmitter(int eid) {   
   delete emitters[eid].emitter;
   emitterUsed[eid] = false;      
}

void EffectsManager :: removeEmitterByTime(int eid,float timeout) {
   emitters[eid].timed = true;
   emitters[eid].timeout = 3.0f;
}

int EffectsManager :: getFirstFreeEmitter() {
   for (int i=0;i<MAX_EMITTER_COUNT;++i)
      if (!emitterUsed[i]) {
         emitterUsed[i] = true;
         return i;
      }

   return -1;
}

void EffectsManager :: updatePlayerPositions(float p1x,float p1y,float p2x,float p2y) {
   px[0] = p1x;
   px[1] = p2x;
   py[0] = p1y;
   py[1] = p2y;
   for (int i=0;i<2;++i) {
      tEmitter &e = emitters[peid[i]];
      e.emitter->setCenter(Vector3d(px[i],py[i],0));
   }
}

void EffectsManager :: setEmitterPos(int eid,float x,float y) {
   emitters[eid].emitter->setCenter(Vector3d(x,y,0));
}

void EffectsManager :: setPlayerDamageEffect(int pid,int level) {
   
   static int table[10] = {-20,-20,-15,-12,-10,-5,0,5,7,10};
   
   if (level >= 0 && level < 10) {            
      tEmitter &e = emitters[peid[pid]];

      if (level == 0)
         e.active = false;
      else
         e.active = true;

      e.emitter->setAlphaScale(-1.0f + 0.1f * table[level]);
   }
}

void EffectsManager :: createPlayerDamageEmitters() {

   for (int i=0;i<2;++i) {

      peid[i] = getFirstFreeEmitter();

      tEmitter &e = emitters[peid[i]];
      e.eid = peid[i];
      e.active = false;
      e.timed = false;
      e.killme = false;

      e.emitter  = new Particle(100);
      e.emitter ->setTexture(yellowSmokeTex);
      e.emitter ->setBlendMode(GL_SRC_ALPHA,GL_ONE_MINUS_SRC_ALPHA);
      const float particleWidth = 0.2f * 0.1f;
      const float particleLength = 1.0f * 0.1f;
      Vector3d center(0,0,0);
      Vector3d minVel(-particleWidth*0.3*0.1,0,0);
      Vector3d maxVel(particleWidth*0.3*0.1,particleLength*0.3,0);
      Vector3d minAcc(0.0f,0.0f,0.0f);
      Vector3d maxAcc(0.0f,0.0f,0.0f);
   
      float _maxr=0.2f,_size=0.01f,_scale_power=0.5f,_life_scale=0.5f,_alpha_scale=-1.0f;
      Vector3d _col_scale(-0.5f,-0.5f,-0.5f);
      e.emitter ->setParameters(center,maxVel,minVel,maxAcc,minAcc,_maxr,_size,_scale_power,_life_scale,_col_scale,_alpha_scale);
      e.emitter ->addRandomEntities(e.emitter->pCount(),center,maxVel,minVel,maxAcc,minAcc,_maxr);
      for (int i=0;i<e.emitter->pCount();++i) {
         e.emitter ->setColor(i,1,0,0,1.0f);
      }
      e.emitter->setGravity(0,0,0);
      e.emitter->setFountain(true);
   
   }
   
}

void EffectsManager :: createPlayerExplosionEmitters() {

   for (int i=0;i<2;++i) {

      pexeid[i] = getFirstFreeEmitter();

      tEmitter &e = emitters[pexeid[i]];
      e.eid = pexeid[i];
      e.active = false;
      e.timed = false;
      e.killme = false;

      e.emitter = new Particle("data/pexp.part");

   }

}

void EffectsManager :: update(float dt) {

   for (int i=0;i<MAX_EMITTER_COUNT;++i)
      if(emitterUsed[i]) {
         
         emitters[i].emitter->update(dt);
         
         if (emitters[i].timed) {
            emitters[i].timeout -= dt;
            if (emitters[i].timeout < 0) {
               removeEmitter(i);
            }
         }
         
      }
     
   DecalList::iterator di = expDecals.begin();
   while (di != expDecals.end()) {
      tExpDecal &decal = *di;
      decal.remaining_life -= decal.life_scale * 0.3f * dt;
      if (decal.remaining_life < 0) {
         DecalList::iterator dn = di;
         ++dn;
         expDecals.erase(di);
         di = dn;
      }
      else
         ++di;
   }

   BloodList::iterator bi = bloodSplats.begin();
   while (bi != bloodSplats.end()) {
      tBloodSplat &decal = *bi;
      decal.remaining_life -= decal.life_scale * 0.3f * dt;
      if (decal.remaining_life < 0) {
         BloodList::iterator dn = bi;
         ++dn;
         bloodSplats.erase(bi);
         bi = dn;
      }
      else
         ++bi;
   }

   ExplosionList::iterator ei = explosions.begin();
   while (ei != explosions.end()) {   
      tExplosion &expl = *ei;

      expl.time -= dt;
      if (expl.time < 0) {
         ExplosionList::iterator itn = ei;
         ++itn;
         freeExpTextures.push_back(expl.tex);
         explosions.erase(ei);
         ei = itn;         
      }
      else
         ++ei;
   }

}

void EffectsManager :: render() {

   glDisable(GL_STENCIL_TEST);
   for (int i=0;i<MAX_EMITTER_COUNT;++i)
      if(emitterUsed[i])
         if (emitters[i].active)
            emitters[i].emitter->render();

   glEnable(GL_BLEND);
   glBlendFunc(GL_ONE_MINUS_DST_ALPHA,GL_ONE_MINUS_SRC_COLOR);
   
   glEnable(GL_STENCIL_TEST);
   DecalList::iterator di = expDecals.begin();
   while (di != expDecals.end()) {
      tExpDecal &decal = *di;
      glPushMatrix();
      glTranslatef(decal.x,decal.y,0);
      glColor3f(decal.remaining_life,decal.remaining_life,decal.remaining_life);
      glScalef(5.0f,5.0f,5.0f);
      expDecalTex->render();
      glPopMatrix();
      ++di;
   }
   
   
   glBlendFunc(GL_ONE,GL_ONE);
   BloodList::iterator bi = bloodSplats.begin();
   while (bi != bloodSplats.end()) {
      tBloodSplat &decal = *bi;
      glPushMatrix();
      glTranslatef(decal.x,decal.y,0);
      glRotatef(decal.rot,0,0,1);
      glColor3f(decal.remaining_life,0,0);
      //glScalef(2.0f,2.0f,2.0f);
      bloodSplatTex->render();
      glPopMatrix();
      ++bi;
   }
   
   
   glDisable(GL_STENCIL_TEST);
   glEnable(GL_BLEND);
   glBlendFunc(GL_ONE,GL_ONE_MINUS_SRC_ALPHA);
   glColor3f(1,1,1);
      
   ExplosionList::iterator ei = explosions.begin();
   while (ei != explosions.end()) {   
      tExplosion &expl = *ei;

      glPushMatrix();
      glTranslatef(expl.x,expl.y,0);
      glScalef(2,2,1);
      expl.tex->render();
      glPopMatrix();

      ++ei;
   }
      
   glEnable(GL_STENCIL_TEST);
   glDisable(GL_BLEND);

}


